/**
 * Admin kullanıcı oluşturma scripti
 * Kullanım: npm run create-admin
 */

const bcrypt = require("bcryptjs");
const mysql = require("mysql2/promise");
const dotenv = require("dotenv");
const path = require("path");

// .env dosyasını yükle
dotenv.config({ path: path.resolve(__dirname, "../.env.local") });

// Veritabanı bağlantısı
const pool = mysql.createPool({
  host: process.env.DB_HOST || "localhost",
  port: parseInt(process.env.DB_PORT || "3306"),
  user: process.env.DB_USER || "root",
  password: process.env.DB_PASSWORD || "",
  database: process.env.DB_NAME || "teknik_servis",
  waitForConnections: true,
  connectionLimit: 10,
  insecureAuth: true,
});

// Şifreyi hash'le
async function hashPassword(password: string): Promise<string> {
  const salt = await bcrypt.genSalt(10);
  return bcrypt.hash(password, salt);
}

// Query çalıştır
async function query(sql: string, params?: any[]) {
  try {
    const [results] = await pool.execute(sql, params);
    return results;
  } catch (error: any) {
    console.error("Veritabanı sorgu hatası:", error);
    throw error;
  }
}

async function createAdminUser() {
  const username = process.env.ADMIN_USERNAME || "admin";
  const password = process.env.ADMIN_PASSWORD || "admin123";
  const email = process.env.ADMIN_EMAIL || "admin@teknikservis.com";
  const fullName = process.env.ADMIN_FULL_NAME || "Admin Kullanıcı";

  try {
    console.log("Admin kullanıcı oluşturuluyor...");

    // Şifreyi hash'le
    const hashedPassword = await hashPassword(password);

    // Kullanıcıyı oluştur veya güncelle
    await query(
      `INSERT INTO users (username, password, email, full_name, role, is_active) 
       VALUES (?, ?, ?, ?, 'admin', TRUE)
       ON DUPLICATE KEY UPDATE 
         password = VALUES(password),
         email = VALUES(email),
         full_name = VALUES(full_name),
         role = 'admin',
         is_active = TRUE`,
      [username, hashedPassword, email, fullName]
    );

    console.log("✅ Admin kullanıcı başarıyla oluşturuldu!");
    console.log(`   Kullanıcı adı: ${username}`);
    console.log(`   Şifre: ${password}`);
    console.log("⚠️  Production'da şifreyi mutlaka değiştirin!");
    
    // Bağlantıyı kapat
    await pool.end();
    process.exit(0);
  } catch (error: any) {
    console.error("❌ Hata:", error.message);
    await pool.end();
    process.exit(1);
  }
}

createAdminUser();
