import { NextRequest, NextResponse } from "next/server";
import { verifyToken } from "./auth";

// Auth middleware - protected routes için
export function authMiddleware(request: NextRequest) {
  const token =
    request.cookies.get("token")?.value ||
    request.headers.get("authorization")?.replace("Bearer ", "");

  if (!token) {
    return {
      authorized: false,
      error: "Token bulunamadı",
    };
  }

  const decoded = verifyToken(token);

  if (!decoded) {
    return {
      authorized: false,
      error: "Geçersiz token",
    };
  }

  return {
    authorized: true,
    user: decoded,
  };
}

// Role-based access control
export function requireRole(allowedRoles: string[]) {
  return (request: NextRequest) => {
    const auth = authMiddleware(request);

    if (!auth.authorized) {
      return {
        authorized: false,
        error: auth.error,
      };
    }

    if (!allowedRoles.includes(auth.user?.role)) {
      return {
        authorized: false,
        error: "Yetkiniz yok",
      };
    }

    return {
      authorized: true,
      user: auth.user,
    };
  };
}

