import jwt from "jsonwebtoken";
import bcrypt from "bcryptjs";
import { query } from "./db";

const JWT_SECRET: string = process.env.JWT_SECRET || "your-secret-key-change-in-production";
const JWT_EXPIRES_IN: string = process.env.JWT_EXPIRES_IN || "7d";

export interface User {
  id: number;
  username: string;
  email: string | null;
  full_name: string | null;
  role: "admin" | "user" | "technician";
  is_active: boolean;
}

export interface LoginCredentials {
  username: string;
  password: string;
}

// Şifreyi hash'le
export async function hashPassword(password: string): Promise<string> {
  const salt = await bcrypt.genSalt(10);
  return bcrypt.hash(password, salt);
}

// Şifreyi doğrula
export async function verifyPassword(
  password: string,
  hashedPassword: string
): Promise<boolean> {
  return bcrypt.compare(password, hashedPassword);
}

// JWT token oluştur
export function generateToken(user: User): string {
  return jwt.sign(
    {
      id: user.id,
      username: user.username,
      role: user.role,
    },
    JWT_SECRET,
    {
      expiresIn: JWT_EXPIRES_IN,
    } as jwt.SignOptions
  );
}

// JWT token doğrula
export function verifyToken(token: string): any {
  try {
    return jwt.verify(token, JWT_SECRET);
  } catch (error) {
    return null;
  }
}

// Kullanıcıyı kullanıcı adı ile bul
export async function findUserByUsername(
  username: string
): Promise<User | null> {
  try {
    const results: any = await query(
      "SELECT id, username, email, full_name, role, is_active, password FROM users WHERE username = ? AND is_active = TRUE",
      [username]
    );

    if (Array.isArray(results) && results.length > 0) {
      const user = results[0];
      // password'u response'dan çıkar
      const { password, ...userWithoutPassword } = user;
      return userWithoutPassword as User;
    }

    return null;
  } catch (error) {
    console.error("Kullanıcı bulunamadı:", error);
    return null;
  }
}

// Kullanıcıyı ID ile bul
export async function findUserById(id: number): Promise<User | null> {
  try {
    const results: any = await query(
      "SELECT id, username, email, full_name, role, is_active FROM users WHERE id = ? AND is_active = TRUE",
      [id]
    );

    if (Array.isArray(results) && results.length > 0) {
      return results[0] as User;
    }

    return null;
  } catch (error) {
    console.error("Kullanıcı bulunamadı:", error);
    return null;
  }
}

// Login işlemi
export async function login(
  credentials: LoginCredentials
): Promise<{ user: User; token: string } | null> {
  try {
    // Kullanıcıyı bul (şifre dahil)
    const results: any = await query(
      "SELECT id, username, email, full_name, role, is_active, password FROM users WHERE username = ? AND is_active = TRUE",
      [credentials.username]
    );

    if (!Array.isArray(results) || results.length === 0) {
      return null;
    }

    const userData = results[0];
    const hashedPassword = userData.password;

    // Şifreyi doğrula
    const isPasswordValid = await verifyPassword(
      credentials.password,
      hashedPassword
    );

    if (!isPasswordValid) {
      return null;
    }

    // Kullanıcı bilgilerini hazırla (şifre olmadan)
    const user: User = {
      id: userData.id,
      username: userData.username,
      email: userData.email,
      full_name: userData.full_name,
      role: userData.role,
      is_active: userData.is_active,
    };

    // Token oluştur
    const token = generateToken(user);

    return { user, token };
  } catch (error) {
    console.error("Login hatası:", error);
    return null;
  }
}

