"use client";

import { useState, useEffect } from "react";

export default function SMSAyarlariPage() {
  const [formData, setFormData] = useState({
    api_id: "",
    api_key: "",
    sms_baslik: "",
    api_url: "https://api.vatansms.net/api/v1",
  });

  const [isSaving, setIsSaving] = useState(false);
  const [isLoading, setIsLoading] = useState(true);
  const [isTesting, setIsTesting] = useState(false);
  const [testType, setTestType] = useState<"senders" | "user_info" | "test_sms" | null>(null);
  const [showTestModal, setShowTestModal] = useState(false);
  const [testPhone, setTestPhone] = useState("");
  const [message, setMessage] = useState<{ type: "success" | "error"; text: string } | null>(null);

  // Veritabanından SMS ayarlarını yükle
  useEffect(() => {
    const fetchSMSAyarlari = async () => {
      try {
        const response = await fetch("/api/ayarlar/sms-ayarlari");
        const data = await response.json();

        if (data.success && data.data) {
          setFormData({
            api_id: data.data.api_id || "",
            api_key: data.data.api_key || "",
            sms_baslik: data.data.sms_baslik || "",
            api_url: data.data.api_url || "https://api.vatansms.net/api/v1",
          });
        }
      } catch (error) {
        console.error("SMS ayarları yükleme hatası:", error);
        setMessage({ type: "error", text: "SMS ayarları yüklenirken bir hata oluştu" });
      } finally {
        setIsLoading(false);
      }
    };

    fetchSMSAyarlari();
  }, []);

  const handleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value } = e.target;
    setFormData((prev) => ({
      ...prev,
      [name]: value,
    }));
  };

  const handleSubmit = async (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    setIsSaving(true);
    setMessage(null);

    try {
      const response = await fetch("/api/ayarlar/sms-ayarlari", {
        method: "PUT",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(formData),
      });

      const data = await response.json();

      if (data.success) {
        setMessage({ type: "success", text: data.message || "SMS ayarları başarıyla güncellendi" });
        // API Key alanını temizle
        setFormData((prev) => ({ ...prev, api_key: "" }));
      } else {
        setMessage({ type: "error", text: data.message || "Güncelleme başarısız oldu" });
      }
    } catch (error) {
      console.error("Güncelleme hatası:", error);
      setMessage({ type: "error", text: "Bir hata oluştu. Lütfen tekrar deneyin." });
    } finally {
      setIsSaving(false);
    }
  };

  const handleTestClick = () => {
    setTestPhone("");
    setShowTestModal(true);
    setMessage(null);
  };


  const handleSendersCheck = async () => {
    setIsTesting(true);
    setTestType("senders");
    setMessage(null);

    try {
      const response = await fetch("/api/ayarlar/sms-ayarlari/test", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          test_type: "senders",
        }),
      });

      const data = await response.json();

      if (data.success) {
        setMessage({ type: "success", text: data.message || "Gönderici adları başarıyla sorgulandı!" });
      } else {
        setMessage({ type: "error", text: data.message || "Gönderici adları sorgulanamadı" });
      }
    } catch (error) {
      console.error("Gönderici adı sorgulama hatası:", error);
      setMessage({ type: "error", text: "Gönderici adı sorgulama sırasında bir hata oluştu. Lütfen tekrar deneyin." });
    } finally {
      setIsTesting(false);
      setTestType(null);
    }
  };

  const handleUserInfoCheck = async () => {
    setIsTesting(true);
    setTestType("user_info");
    setMessage(null);

    try {
      const response = await fetch("/api/ayarlar/sms-ayarlari/test", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          test_type: "user_info",
        }),
      });

      const data = await response.json();

      if (data.success) {
        setMessage({ type: "success", text: data.message || "Kullanıcı bilgileri başarıyla sorgulandı!" });
      } else {
        setMessage({ type: "error", text: data.message || "Kullanıcı bilgileri sorgulanamadı" });
      }
    } catch (error) {
      console.error("Kullanıcı bilgileri sorgulama hatası:", error);
      setMessage({ type: "error", text: "Kullanıcı bilgileri sorgulama sırasında bir hata oluştu. Lütfen tekrar deneyin." });
    } finally {
      setIsTesting(false);
      setTestType(null);
    }
  };

  const handleTest = async () => {
    if (!testPhone || testPhone.trim() === "") {
      setMessage({ type: "error", text: "Lütfen test SMS gönderilecek telefon numarasını giriniz." });
      return;
    }

    // Telefon format kontrolü
    const phoneRegex = /^[0-9\s()+-]+$/;
    if (!phoneRegex.test(testPhone)) {
      setMessage({ type: "error", text: "Lütfen geçerli bir telefon numarası giriniz." });
      return;
    }

    setIsTesting(true);
    setTestType("test_sms");
    setMessage(null);

    try {
      const response = await fetch("/api/ayarlar/sms-ayarlari/test", {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          test_type: "test_sms",
          test_phone: testPhone,
        }),
      });

      const data = await response.json();

      if (data.success) {
        setMessage({ type: "success", text: data.message || "Test SMS başarıyla gönderildi!" });
        setShowTestModal(false);
      } else {
        setMessage({ type: "error", text: data.message || "Test SMS gönderilemedi" });
      }
    } catch (error) {
      console.error("Test hatası:", error);
      setMessage({ type: "error", text: "Test sırasında bir hata oluştu. Lütfen tekrar deneyin." });
    } finally {
      setIsTesting(false);
      setTestType(null);
    }
  };

  if (isLoading) {
    return (
      <div className="space-y-6">
        <div className="bg-white dark:bg-gray-800 rounded-lg shadow p-6">
          <div className="flex items-center justify-center py-12">
            <div className="text-center">
              <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-indigo-600 mx-auto"></div>
              <p className="mt-4 text-gray-600 dark:text-gray-400">Yükleniyor...</p>
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="bg-white dark:bg-gray-800 rounded-lg shadow">
        {/* Header */}
        <div className="flex items-center justify-between px-6 py-4 border-b border-gray-200 dark:border-gray-700">
          <h2 className="text-xl font-bold text-gray-900 dark:text-white">
            SMS Ayarları (Vatan SMS)
          </h2>
          <button className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300">
            <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 8V4m0 0h4M4 4l5 5m11-1V4m0 0h-4m4 0l-5 5M4 16v4m0 0h4m-4 0l5-5m11 5l-5-5m5 5v-4m0 4h-4" />
            </svg>
          </button>
        </div>

        {/* Message */}
        {message && (
          <div
            className={`mx-6 mt-4 px-4 py-3 rounded-lg flex items-center space-x-3 ${
              message.type === "success"
                ? "bg-green-50 dark:bg-green-900/20 text-green-600 dark:text-green-400 border border-green-200 dark:border-green-800"
                : "bg-red-50 dark:bg-red-900/20 text-red-600 dark:text-red-400 border border-red-200 dark:border-red-800"
            }`}
          >
            {message.type === "success" ? (
              <svg className="w-5 h-5 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
            ) : (
              <svg className="w-5 h-5 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10 14l2-2m0 0l2-2m-2 2l-2-2m2 2l2 2m7-2a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
            )}
            <span>{message.text}</span>
          </div>
        )}

        {/* Form */}
        <form onSubmit={handleSubmit} className="p-6">
          <div className="space-y-4 max-w-4xl">
            {/* İlk Satır: API ID ve API Key */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              {/* API ID */}
              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  API ID <span className="text-red-500">*</span>
                </label>
                <input
                  type="text"
                  name="api_id"
                  value={formData.api_id}
                  onChange={handleChange}
                  required
                  placeholder="Vatan SMS API ID"
                  className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                />
                <p className="mt-2 text-sm text-gray-500 dark:text-gray-400">
                  Vatan SMS panelinizden alacağınız API ID
                </p>
              </div>

              {/* API Key */}
              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  API Key
                </label>
                <input
                  type="text"
                  name="api_key"
                  value={formData.api_key}
                  onChange={handleChange}
                  placeholder="Vatan SMS API Key"
                  className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                />
                <p className="mt-2 text-sm text-orange-600 dark:text-orange-400">
                  API Key&apos;i değiştirmek istemiyorsanız boş bırakınız.
                </p>
              </div>
            </div>

            {/* İkinci Satır: SMS Başlığı ve API URL */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              {/* SMS Başlığı */}
              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  SMS Başlığı (Sender)
                </label>
                <input
                  type="text"
                  name="sms_baslik"
                  value={formData.sms_baslik}
                  onChange={handleChange}
                  placeholder="GONDERICI_ADI"
                  maxLength={11}
                  className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                />
                <p className="mt-2 text-sm text-gray-500 dark:text-gray-400">
                  Onaylanmış SMS başlığınız (maksimum 11 karakter)
                </p>
              </div>

              {/* API URL */}
              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  API URL
                </label>
                <input
                  type="text"
                  name="api_url"
                  value={formData.api_url}
                  onChange={handleChange}
                  placeholder="https://api.vatansms.net/api/v1"
                  className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                />
              </div>
            </div>
          </div>

          {/* Buttons */}
          <div className="mt-6 flex items-center space-x-4 flex-wrap gap-4">
            <button
              type="submit"
              disabled={isSaving || isTesting}
              className="bg-green-600 hover:bg-green-700 text-white font-semibold px-6 py-2 rounded-lg transition-colors duration-200 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {isSaving ? "Güncelleniyor..." : "Güncelle"}
            </button>
            <button
              type="button"
              onClick={handleSendersCheck}
              disabled={isSaving || isTesting || !formData.api_id || !formData.api_key}
              className="bg-indigo-600 hover:bg-indigo-700 text-white font-semibold px-6 py-2 rounded-lg transition-colors duration-200 disabled:opacity-50 disabled:cursor-not-allowed flex items-center space-x-2"
            >
              {isTesting && testType === "senders" ? (
                <>
                  <svg className="animate-spin h-4 w-4 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                    <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                    <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                  </svg>
                  <span>Sorgulanıyor...</span>
                </>
              ) : (
                <>
                  <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M7 8h10M7 12h4m1 8l-4-4H5a2 2 0 01-2-2V6a2 2 0 012-2h14a2 2 0 012 2v8a2 2 0 01-2 2h-3l-4 4z" />
                  </svg>
                  <span>Gönderici Adı Sorgula</span>
                </>
              )}
            </button>
            <button
              type="button"
              onClick={handleUserInfoCheck}
              disabled={isSaving || isTesting || !formData.api_id || !formData.api_key}
              className="bg-teal-600 hover:bg-teal-700 text-white font-semibold px-6 py-2 rounded-lg transition-colors duration-200 disabled:opacity-50 disabled:cursor-not-allowed flex items-center space-x-2"
            >
              {isTesting && testType === "user_info" ? (
                <>
                  <svg className="animate-spin h-4 w-4 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                    <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                    <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                  </svg>
                  <span>Sorgulanıyor...</span>
                </>
              ) : (
                <>
                  <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z" />
                  </svg>
                  <span>Kullanıcı Bilgileri</span>
                </>
              )}
            </button>
            <button
              type="button"
              onClick={handleTestClick}
              disabled={isSaving || isTesting || !formData.api_id || !formData.api_key}
              className="bg-blue-600 hover:bg-blue-700 text-white font-semibold px-6 py-2 rounded-lg transition-colors duration-200 disabled:opacity-50 disabled:cursor-not-allowed flex items-center space-x-2"
            >
              <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 18h.01M8 21h8a2 2 0 002-2V5a2 2 0 00-2-2H8a2 2 0 00-2 2v14a2 2 0 002 2z" />
              </svg>
              <span>Test SMS Gönder</span>
            </button>
          </div>
        </form>
      </div>

      {/* Test Modal */}
      {showTestModal && (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-black bg-opacity-50">
          <div className="bg-white dark:bg-gray-800 rounded-lg shadow-xl max-w-md w-full mx-4">
            <div className="p-6">
              {/* Header */}
              <div className="flex items-center justify-between mb-4">
                <h3 className="text-xl font-bold text-gray-900 dark:text-white">
                  Test SMS Gönder
                </h3>
                <button
                  onClick={() => {
                    setShowTestModal(false);
                    setMessage(null);
                  }}
                  className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
                  disabled={isTesting}
                >
                  <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                  </svg>
                </button>
              </div>

              {/* Message */}
              {message && (
                <div
                  className={`mb-4 px-4 py-3 rounded-lg flex items-center space-x-3 ${
                    message.type === "success"
                      ? "bg-green-50 dark:bg-green-900/20 text-green-600 dark:text-green-400 border border-green-200 dark:border-green-800"
                      : "bg-red-50 dark:bg-red-900/20 text-red-600 dark:text-red-400 border border-red-200 dark:border-red-800"
                  }`}
                >
                  {message.type === "success" ? (
                    <svg className="w-5 h-5 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
                    </svg>
                  ) : (
                    <svg className="w-5 h-5 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10 14l2-2m0 0l2-2m-2 2l-2-2m2 2l2 2m7-2a9 9 0 11-18 0 9 9 0 0118 0z" />
                    </svg>
                  )}
                  <span className="text-sm">{message.text}</span>
                </div>
              )}

              {/* Form */}
              <div className="space-y-4">
                <div>
                  <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                    Test Telefon Numarası <span className="text-red-500">*</span>
                  </label>
                  <input
                    type="tel"
                    value={testPhone}
                    onChange={(e) => setTestPhone(e.target.value)}
                    placeholder="905XXXXXXXXX veya 0XXX XXX XX XX"
                    disabled={isTesting}
                    className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white disabled:opacity-50 disabled:cursor-not-allowed"
                  />
                  <p className="mt-2 text-sm text-gray-500 dark:text-gray-400">
                    Test SMS bu numaraya gönderilecektir. (90 ile başlamalı veya 0 ile başlayabilir)
                  </p>
                </div>
              </div>

              {/* Buttons */}
              <div className="mt-6 flex items-center justify-end space-x-3">
                <button
                  type="button"
                  onClick={() => {
                    setShowTestModal(false);
                    setMessage(null);
                  }}
                  disabled={isTesting}
                  className="px-4 py-2 text-gray-700 dark:text-gray-300 bg-gray-100 dark:bg-gray-700 rounded-lg hover:bg-gray-200 dark:hover:bg-gray-600 transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
                >
                  İptal
                </button>
                <button
                  type="button"
                  onClick={handleTest}
                  disabled={isTesting || !testPhone}
                  className="px-6 py-2 bg-blue-600 hover:bg-blue-700 text-white font-semibold rounded-lg transition-colors disabled:opacity-50 disabled:cursor-not-allowed flex items-center space-x-2"
                >
                  {isTesting ? (
                    <>
                      <svg className="animate-spin h-4 w-4 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                        <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                        <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                      </svg>
                      <span>Gönderiliyor...</span>
                    </>
                  ) : (
                    <>
                      <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 19l9 2-9-18-9 18 9-2zm0 0v-8" />
                      </svg>
                      <span>Gönder</span>
                    </>
                  )}
                </button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
