"use client";

import { useState, useEffect } from "react";

export default function PanelAyarlariPage() {
  const [formData, setFormData] = useState({
    panel_adi: "",
    logo: "",
    stok_uyari_limiti: "",
    izin_verilen_dosya_turleri: "",
    hizmet_kdv_orani: "",
    urun_kdv_orani: "",
    para_birimi: "TRY",
  });

  const [isSaving, setIsSaving] = useState(false);
  const [isLoading, setIsLoading] = useState(true);
  const [logoPreview, setLogoPreview] = useState<string | null>(null);
  const [message, setMessage] = useState<{ type: "success" | "error"; text: string } | null>(null);
  const [dosyaTurleri, setDosyaTurleri] = useState<string[]>([]);
  const [yeniDosyaTuru, setYeniDosyaTuru] = useState("");

  // Veritabanından panel ayarlarını yükle
  useEffect(() => {
    const fetchPanelAyarlari = async () => {
      try {
        const response = await fetch("/api/ayarlar/panel-ayarlari");
        const data = await response.json();

        if (data.success && data.data) {
          setFormData({
            panel_adi: data.data.panel_adi || "",
            logo: data.data.logo || "",
            stok_uyari_limiti: data.data.stok_uyari_limiti || "",
            izin_verilen_dosya_turleri: data.data.izin_verilen_dosya_turleri || "",
            hizmet_kdv_orani: data.data.hizmet_kdv_orani || "",
            urun_kdv_orani: data.data.urun_kdv_orani || "",
            para_birimi: data.data.para_birimi || "TRY",
          });
          if (data.data.logo) {
            // Eğer logo bir URL/path ise direkt kullan, base64 ise olduğu gibi
            setLogoPreview(data.data.logo.startsWith("data:") ? data.data.logo : data.data.logo);
          }
          // Dosya türlerini array'e çevir
          if (data.data.izin_verilen_dosya_turleri) {
            const turler = data.data.izin_verilen_dosya_turleri
              .split(",")
              .map((t: string) => t.trim())
              .filter((t: string) => t.length > 0);
            setDosyaTurleri(turler);
          } else {
            setDosyaTurleri([]);
          }
        }
      } catch (error) {
        console.error("Panel ayarları yükleme hatası:", error);
        setMessage({ type: "error", text: "Panel ayarları yüklenirken bir hata oluştu" });
      } finally {
        setIsLoading(false);
      }
    };

    fetchPanelAyarlari();
  }, []);

  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement | HTMLTextAreaElement>) => {
    const { name, value } = e.target;
    setFormData((prev) => ({
      ...prev,
      [name]: value,
    }));
  };

  const handleLogoChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (file) {
      // Dosya türü kontrolü
      const allowedTypes = ["image/jpeg", "image/jpg", "image/png", "image/gif", "image/svg+xml"];
      if (!allowedTypes.includes(file.type)) {
        setMessage({ type: "error", text: "Sadece resim dosyaları yüklenebilir (JPEG, PNG, GIF, SVG)" });
        return;
      }

      // Dosya boyutu kontrolü (5MB)
      if (file.size > 5 * 1024 * 1024) {
        setMessage({ type: "error", text: "Dosya boyutu 5MB'dan küçük olmalıdır" });
        return;
      }

      // Preview oluştur
      const reader = new FileReader();
      reader.onloadend = () => {
        setLogoPreview(reader.result as string);
      };
      reader.readAsDataURL(file);

      // FormData'ya ekle (backend'e gönderilecek)
      setFormData((prev) => ({
        ...prev,
        logo: file.name,
      }));
    }
  };

  const handleDosyaTuruEkle = () => {
    const tur = yeniDosyaTuru.trim().toLowerCase();
    if (tur && !dosyaTurleri.includes(tur)) {
      setDosyaTurleri([...dosyaTurleri, tur]);
      setYeniDosyaTuru("");
    }
  };

  const handleDosyaTuruSil = (index: number) => {
    setDosyaTurleri(dosyaTurleri.filter((_, i) => i !== index));
  };

  const handleDosyaTuruKeyPress = (e: React.KeyboardEvent<HTMLInputElement>) => {
    if (e.key === "Enter") {
      e.preventDefault();
      handleDosyaTuruEkle();
    }
  };

  const handleSubmit = async (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    setIsSaving(true);
    setMessage(null);

    try {
      const formDataToSend = new FormData();
      Object.keys(formData).forEach((key) => {
        if (key !== "logo" && key !== "izin_verilen_dosya_turleri") {
          formDataToSend.append(key, formData[key as keyof typeof formData]);
        }
      });

      // Dosya türlerini virgülle ayrılmış string olarak ekle
      const dosyaTurleriString = dosyaTurleri.join(",");
      formDataToSend.append("izin_verilen_dosya_turleri", dosyaTurleriString);

      // Logo dosyasını ekle
      const logoInput = document.querySelector<HTMLInputElement>('input[name="logo_file"]');
      if (logoInput?.files?.[0]) {
        formDataToSend.append("logo_file", logoInput.files[0]);
      } else if (formData.logo) {
        formDataToSend.append("logo", formData.logo);
      }

      const response = await fetch("/api/ayarlar/panel-ayarlari", {
        method: "PUT",
        body: formDataToSend,
      });

      const data = await response.json();

      if (data.success) {
        setMessage({ type: "success", text: data.message || "Panel ayarları başarıyla güncellendi" });
      } else {
        setMessage({ type: "error", text: data.message || "Güncelleme başarısız oldu" });
      }
    } catch (error) {
      console.error("Güncelleme hatası:", error);
      setMessage({ type: "error", text: "Bir hata oluştu. Lütfen tekrar deneyin." });
    } finally {
      setIsSaving(false);
    }
  };

  if (isLoading) {
    return (
      <div className="space-y-6">
        <div className="bg-white dark:bg-gray-800 rounded-lg shadow p-6">
          <div className="flex items-center justify-center py-12">
            <div className="text-center">
              <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-indigo-600 mx-auto"></div>
              <p className="mt-4 text-gray-600 dark:text-gray-400">Yükleniyor...</p>
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="bg-white dark:bg-gray-800 rounded-lg shadow">
        {/* Header */}
        <div className="flex items-center justify-between px-6 py-4 border-b border-gray-200 dark:border-gray-700">
          <h2 className="text-xl font-bold text-gray-900 dark:text-white">
            Panel Ayarları
          </h2>
          <button className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300">
            <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 8V4m0 0h4M4 4l5 5m11-1V4m0 0h-4m4 0l-5 5M4 16v4m0 0h4m-4 0l5-5m11 5l-5-5m5 5v-4m0 4h-4" />
            </svg>
          </button>
        </div>

        {/* Message */}
        {message && (
          <div
            className={`mx-6 mt-4 px-4 py-3 rounded-lg flex items-center space-x-3 ${
              message.type === "success"
                ? "bg-green-50 dark:bg-green-900/20 text-green-600 dark:text-green-400 border border-green-200 dark:border-green-800"
                : "bg-red-50 dark:bg-red-900/20 text-red-600 dark:text-red-400 border border-red-200 dark:border-red-800"
            }`}
          >
            {message.type === "success" ? (
              <svg className="w-5 h-5 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
            ) : (
              <svg className="w-5 h-5 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10 14l2-2m0 0l2-2m-2 2l-2-2m2 2l2 2m7-2a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
            )}
            <span>{message.text}</span>
          </div>
        )}

        {/* Form */}
        <form onSubmit={handleSubmit} className="p-6">
          <div className="space-y-4 max-w-4xl">
            {/* İlk Satır: Panel Adı ve Logo */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              {/* Panel Adı */}
              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  Panel Adı <span className="text-red-500">*</span>
                </label>
                <input
                  type="text"
                  name="panel_adi"
                  value={formData.panel_adi}
                  onChange={handleChange}
                  required
                  className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                />
              </div>

              {/* Logo */}
              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  Logo
                </label>
                <div className="space-y-2">
                  <input
                    type="file"
                    name="logo_file"
                    accept="image/jpeg,image/jpg,image/png,image/gif,image/svg+xml"
                    onChange={handleLogoChange}
                    className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:text-sm file:font-semibold file:bg-indigo-50 file:text-indigo-700 hover:file:bg-indigo-100 dark:file:bg-indigo-900/20 dark:file:text-indigo-300"
                  />
                  {logoPreview && (
                    <div className="mt-2">
                      <img
                        src={logoPreview.startsWith("data:") ? logoPreview : logoPreview}
                        alt="Logo önizleme"
                        className="h-20 w-auto object-contain border border-gray-300 dark:border-gray-600 rounded-lg p-2 bg-gray-50 dark:bg-gray-700"
                      />
                    </div>
                  )}
                </div>
              </div>
            </div>

            {/* İkinci Satır: Stok Uyarı Limiti ve İzin Verilen Dosya Türleri */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              {/* Stok Uyarı Limiti */}
              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  Stok Uyarı Limiti
                </label>
                <input
                  type="number"
                  name="stok_uyari_limiti"
                  value={formData.stok_uyari_limiti}
                  onChange={handleChange}
                  min="0"
                  step="1"
                  className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                />
              </div>

              {/* İzin Verilen Dosya Türleri */}
              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  İzin Verilen Dosya Türleri
                </label>
                <div className="space-y-2">
                  {/* Tag'ler */}
                  {dosyaTurleri.length > 0 && (
                    <div className="flex flex-wrap gap-2 p-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-gray-50 dark:bg-gray-700/50 min-h-[50px]">
                      {dosyaTurleri.map((tur, index) => (
                        <span
                          key={index}
                          className="inline-flex items-center gap-1 px-3 py-1 bg-indigo-100 dark:bg-indigo-900/30 text-indigo-800 dark:text-indigo-200 rounded-full text-sm font-medium"
                        >
                          {tur}
                          <button
                            type="button"
                            onClick={() => handleDosyaTuruSil(index)}
                            className="ml-1 text-indigo-600 dark:text-indigo-300 hover:text-indigo-800 dark:hover:text-indigo-100 focus:outline-none"
                          >
                            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                            </svg>
                          </button>
                        </span>
                      ))}
                    </div>
                  )}
                  {/* Input ve Ekle Butonu */}
                  <div className="flex gap-2">
                    <input
                      type="text"
                      value={yeniDosyaTuru}
                      onChange={(e) => setYeniDosyaTuru(e.target.value)}
                      onKeyPress={handleDosyaTuruKeyPress}
                      placeholder="Dosya türü girin (örn: jpg)"
                      className="flex-1 px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                    />
                    <button
                      type="button"
                      onClick={handleDosyaTuruEkle}
                      disabled={!yeniDosyaTuru.trim() || dosyaTurleri.includes(yeniDosyaTuru.trim().toLowerCase())}
                      className="px-4 py-2 bg-indigo-600 hover:bg-indigo-700 text-white font-medium rounded-lg transition-colors duration-200 disabled:opacity-50 disabled:cursor-not-allowed"
                    >
                      Ekle
                    </button>
                  </div>
                  <p className="text-xs text-gray-500 dark:text-gray-400">
                    Dosya türü girin ve Enter&apos;a basın veya Ekle butonuna tıklayın
                  </p>
                </div>
              </div>
            </div>

            {/* Üçüncü Satır: Hizmet K.D.V. Oranı ve Ürün K.D.V. Oranı */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              {/* Hizmet K.D.V. Oranı */}
              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  Hizmet K.D.V. Oranı (%)
                </label>
                <input
                  type="number"
                  name="hizmet_kdv_orani"
                  value={formData.hizmet_kdv_orani}
                  onChange={handleChange}
                  min="0"
                  max="100"
                  step="0.01"
                  placeholder="Örn: 20"
                  className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                />
              </div>

              {/* Ürün K.D.V. Oranı */}
              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  Ürün K.D.V. Oranı (%)
                </label>
                <input
                  type="number"
                  name="urun_kdv_orani"
                  value={formData.urun_kdv_orani}
                  onChange={handleChange}
                  min="0"
                  max="100"
                  step="0.01"
                  placeholder="Örn: 20"
                  className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                />
              </div>
            </div>

            {/* Dördüncü Satır: Para Birimi */}
            <div>
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                Para Birimi <span className="text-red-500">*</span>
              </label>
              <select
                name="para_birimi"
                value={formData.para_birimi}
                onChange={handleChange}
                required
                className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
              >
                <option value="TRY">TRY - Türk Lirası</option>
                <option value="USD">USD - Amerikan Doları</option>
                <option value="EUR">EUR - Euro</option>
                <option value="GBP">GBP - İngiliz Sterlini</option>
                <option value="JPY">JPY - Japon Yeni</option>
                <option value="CNY">CNY - Çin Yuanı</option>
                <option value="RUB">RUB - Rus Rublesi</option>
              </select>
            </div>
          </div>

          {/* Submit Button */}
          <div className="mt-6">
            <button
              type="submit"
              disabled={isSaving}
              className="bg-green-600 hover:bg-green-700 text-white font-semibold px-6 py-2 rounded-lg transition-colors duration-200 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {isSaving ? "Güncelleniyor..." : "Güncelle"}
            </button>
          </div>
        </form>
      </div>
    </div>
  );
}
