"use client";

import { useState, useEffect } from "react";

export default function FirmaBilgileriPage() {
  const [formData, setFormData] = useState({
    firma_adi: "",
    firma_yetkilisi: "",
    telefon_1: "",
    telefon_2: "",
    email: "",
    adres: "",
    site_adresi: "",
    banka_adi: "",
    banka_hesap_sahibi_adi: "",
    iban_adresi: "",
  });

  const [isSaving, setIsSaving] = useState(false);
  const [isLoading, setIsLoading] = useState(true);
  const [message, setMessage] = useState<{ type: "success" | "error"; text: string } | null>(null);

  // Veritabanından firma bilgilerini yükle
  useEffect(() => {
    const fetchFirmaBilgileri = async () => {
      try {
        const response = await fetch("/api/ayarlar/firma-bilgileri");
        const data = await response.json();

        if (data.success && data.data) {
          // Telefon numaralarını formatla
          const formatPhone = (phone: string) => {
            if (!phone) return "";
            return formatPhoneNumber(phone);
          };

          setFormData({
            firma_adi: data.data.firma_adi || "",
            firma_yetkilisi: data.data.firma_yetkilisi || "",
            telefon_1: formatPhone(data.data.telefon_1 || ""),
            telefon_2: formatPhone(data.data.telefon_2 || ""),
            email: data.data.email || "",
            adres: data.data.adres || "",
            site_adresi: data.data.site_adresi || "",
            banka_adi: data.data.banka_adi || "",
            banka_hesap_sahibi_adi: data.data.banka_hesap_sahibi_adi || "",
            iban_adresi: data.data.iban_adresi || "",
          });
        }
      } catch (error) {
        console.error("Firma bilgileri yükleme hatası:", error);
        setMessage({ type: "error", text: "Firma bilgileri yüklenirken bir hata oluştu" });
      } finally {
        setIsLoading(false);
      }
    };

    fetchFirmaBilgileri();
  }, []);

  // Telefon numarasını (xxx) xxx xxxx formatına çevir
  const formatPhoneNumber = (value: string): string => {
    // Sadece rakamları al
    const numbers = value.replace(/\D/g, "");
    
    // 10 haneden fazla rakam varsa kes
    const limitedNumbers = numbers.slice(0, 10);
    
    // Formatla: (xxx) xxx xxxx
    if (limitedNumbers.length === 0) return "";
    if (limitedNumbers.length <= 3) return `(${limitedNumbers}`;
    if (limitedNumbers.length <= 6) return `(${limitedNumbers.slice(0, 3)}) ${limitedNumbers.slice(3)}`;
    return `(${limitedNumbers.slice(0, 3)}) ${limitedNumbers.slice(3, 6)} ${limitedNumbers.slice(6)}`;
  };

  // Telefon numarasını veritabanına kaydetmek için temizle (sadece rakamlar)
  const cleanPhoneNumber = (value: string): string => {
    return value.replace(/\D/g, "");
  };

  const handleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value } = e.target;
    
    // Telefon alanları için maske uygula
    if (name === "telefon_1" || name === "telefon_2") {
      const formatted = formatPhoneNumber(value);
      setFormData((prev) => ({
        ...prev,
        [name]: formatted,
      }));
    } else {
      setFormData((prev) => ({
        ...prev,
        [name]: value,
      }));
    }
  };

  const handleSubmit = async (e: React.FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    setIsSaving(true);
    setMessage(null);

    try {
      // Telefon numaralarını temizle (sadece rakamlar)
      const submitData = {
        ...formData,
        telefon_1: cleanPhoneNumber(formData.telefon_1),
        telefon_2: cleanPhoneNumber(formData.telefon_2),
      };

      const response = await fetch("/api/ayarlar/firma-bilgileri", {
        method: "PUT",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(submitData),
      });

      const data = await response.json();

      if (data.success) {
        setMessage({ type: "success", text: data.message || "Firma bilgileri başarıyla güncellendi" });
      } else {
        setMessage({ type: "error", text: data.message || "Güncelleme başarısız oldu" });
      }
    } catch (error) {
      console.error("Güncelleme hatası:", error);
      setMessage({ type: "error", text: "Bir hata oluştu. Lütfen tekrar deneyin." });
    } finally {
      setIsSaving(false);
    }
  };

  if (isLoading) {
    return (
      <div className="space-y-6">
        <div className="bg-white dark:bg-gray-800 rounded-lg shadow p-6">
          <div className="flex items-center justify-center py-12">
            <div className="text-center">
              <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-indigo-600 mx-auto"></div>
              <p className="mt-4 text-gray-600 dark:text-gray-400">Yükleniyor...</p>
            </div>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div className="space-y-6">
      <div className="bg-white dark:bg-gray-800 rounded-lg shadow">
        {/* Header */}
        <div className="flex items-center justify-between px-6 py-4 border-b border-gray-200 dark:border-gray-700">
          <h2 className="text-xl font-bold text-gray-900 dark:text-white">
            Firma Bilgileri
          </h2>
          <button className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300">
            <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 8V4m0 0h4M4 4l5 5m11-1V4m0 0h-4m4 0l-5 5M4 16v4m0 0h4m-4 0l5-5m11 5l-5-5m5 5v-4m0 4h-4" />
            </svg>
          </button>
        </div>

        {/* Message */}
        {message && (
          <div
            className={`mx-6 mt-4 px-4 py-3 rounded-lg flex items-center space-x-3 ${
              message.type === "success"
                ? "bg-green-50 dark:bg-green-900/20 text-green-600 dark:text-green-400 border border-green-200 dark:border-green-800"
                : "bg-red-50 dark:bg-red-900/20 text-red-600 dark:text-red-400 border border-red-200 dark:border-red-800"
            }`}
          >
            {message.type === "success" ? (
              <svg className="w-5 h-5 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
            ) : (
              <svg className="w-5 h-5 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10 14l2-2m0 0l2-2m-2 2l-2-2m2 2l2 2m7-2a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
            )}
            <span>{message.text}</span>
          </div>
        )}

        {/* Form */}
        <form onSubmit={handleSubmit} className="p-6">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
            {/* Sol Sütun */}
            <div className="space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  Firma Adı <span className="text-red-500">*</span>
                </label>
                <input
                  type="text"
                  name="firma_adi"
                  value={formData.firma_adi}
                  onChange={handleChange}
                  required
                  className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  Telefon 2
                </label>
                <input
                  type="text"
                  name="telefon_2"
                  value={formData.telefon_2}
                  onChange={handleChange}
                  placeholder="(555) 123 4567"
                  maxLength={14}
                  className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  Site Adresi
                </label>
                <input
                  type="text"
                  name="site_adresi"
                  value={formData.site_adresi}
                  onChange={handleChange}
                  className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  İban Adresi
                </label>
                <input
                  type="text"
                  name="iban_adresi"
                  value={formData.iban_adresi}
                  onChange={handleChange}
                  className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                />
              </div>
            </div>

            {/* Orta Sütun */}
            <div className="space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  Firma Yetkilisi
                </label>
                <input
                  type="text"
                  name="firma_yetkilisi"
                  value={formData.firma_yetkilisi}
                  onChange={handleChange}
                  className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  Email
                </label>
                <input
                  type="email"
                  name="email"
                  value={formData.email}
                  onChange={handleChange}
                  className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  Banka Hesap Sahibi Adı
                </label>
                <input
                  type="text"
                  name="banka_hesap_sahibi_adi"
                  value={formData.banka_hesap_sahibi_adi}
                  onChange={handleChange}
                  className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                />
              </div>
            </div>

            {/* Sağ Sütun */}
            <div className="space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  Telefon 1
                </label>
                <input
                  type="text"
                  name="telefon_1"
                  value={formData.telefon_1}
                  onChange={handleChange}
                  placeholder="(555) 123 4567"
                  maxLength={14}
                  className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  Adres
                </label>
                <input
                  type="text"
                  name="adres"
                  value={formData.adres}
                  onChange={handleChange}
                  className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                />
              </div>

              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  Banka Adı
                </label>
                <input
                  type="text"
                  name="banka_adi"
                  value={formData.banka_adi}
                  onChange={handleChange}
                  className="w-full px-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 focus:border-transparent outline-none bg-white dark:bg-gray-700 text-gray-900 dark:text-white"
                />
              </div>
            </div>
          </div>

          {/* Submit Button */}
          <div className="mt-6">
            <button
              type="submit"
              disabled={isSaving}
              className="bg-green-600 hover:bg-green-700 text-white font-semibold px-6 py-2 rounded-lg transition-colors duration-200 disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {isSaving ? "Güncelleniyor..." : "Güncelle"}
            </button>
          </div>
        </form>
      </div>
    </div>
  );
}
