import { NextRequest, NextResponse } from "next/server";
import { query } from "@/lib/db";
import { verifyToken, hashPassword } from "@/lib/auth";

// Yönetici bilgilerini getir (giriş yapmış kullanıcının)
export async function GET(request: NextRequest) {
  try {
    // Token'ı kontrol et
    const token =
      request.cookies.get("token")?.value ||
      request.headers.get("authorization")?.replace("Bearer ", "");

    if (!token) {
      return NextResponse.json(
        {
          success: false,
          message: "Token bulunamadı",
        },
        { status: 401 }
      );
    }

    const decoded = verifyToken(token);

    if (!decoded || !decoded.id) {
      return NextResponse.json(
        {
          success: false,
          message: "Geçersiz token",
        },
        { status: 401 }
      );
    }

    // Kullanıcıyı bul
    const results: any = await query(
      "SELECT id, username, email, full_name, telefon FROM users WHERE id = ? AND is_active = TRUE",
      [decoded.id]
    );

    if (Array.isArray(results) && results.length > 0) {
      const user = results[0];
      
      // Telefon numarasını formatla (eğer varsa)
      let telefon = user.telefon || "";
      if (telefon && !telefon.includes("(")) {
        // Formatla: (xxx) xxx xxxx
        const numbers = telefon.replace(/\D/g, "");
        if (numbers.length === 10) {
          telefon = `(${numbers.slice(0, 3)}) ${numbers.slice(3, 6)} ${numbers.slice(6)}`;
        }
      }

      return NextResponse.json(
        {
          success: true,
          data: {
            ad_soyad: user.full_name || "",
            telefon: telefon,
            email: user.email || "",
            kullanici_adi: user.username || "",
          },
        },
        { status: 200 }
      );
    }

    return NextResponse.json(
      {
        success: false,
        message: "Kullanıcı bulunamadı",
      },
      { status: 404 }
    );
  } catch (error: any) {
    console.error("Yönetici bilgileri getirme hatası:", error);
    return NextResponse.json(
      {
        success: false,
        message: "Yönetici bilgileri alınamadı",
        error: error.message,
      },
      { status: 500 }
    );
  }
}

// Yönetici bilgilerini güncelle
export async function PUT(request: NextRequest) {
  try {
    // Token'ı kontrol et
    const token =
      request.cookies.get("token")?.value ||
      request.headers.get("authorization")?.replace("Bearer ", "");

    if (!token) {
      return NextResponse.json(
        {
          success: false,
          message: "Token bulunamadı",
        },
        { status: 401 }
      );
    }

    const decoded = verifyToken(token);

    if (!decoded || !decoded.id) {
      return NextResponse.json(
        {
          success: false,
          message: "Geçersiz token",
        },
        { status: 401 }
      );
    }

    const body = await request.json();
    const { ad_soyad, telefon, email, kullanici_adi, sifre } = body;

    // Validasyon
    if (!ad_soyad || ad_soyad.trim() === "") {
      return NextResponse.json(
        {
          success: false,
          message: "Ad Soyad zorunludur",
        },
        { status: 400 }
      );
    }

    if (!email || email.trim() === "") {
      return NextResponse.json(
        {
          success: false,
          message: "Email zorunludur",
        },
        { status: 400 }
      );
    }

    if (!kullanici_adi || kullanici_adi.trim() === "") {
      return NextResponse.json(
        {
          success: false,
          message: "Kullanıcı adı zorunludur",
        },
        { status: 400 }
      );
    }

    // Kullanıcı adı kontrolü (başka kullanıcıda varsa)
    const existingUser: any = await query(
      "SELECT id FROM users WHERE username = ? AND id != ?",
      [kullanici_adi, decoded.id]
    );

    if (Array.isArray(existingUser) && existingUser.length > 0) {
      return NextResponse.json(
        {
          success: false,
          message: "Bu kullanıcı adı zaten kullanılıyor",
        },
        { status: 400 }
      );
    }

    // Email kontrolü (başka kullanıcıda varsa)
    const existingEmail: any = await query(
      "SELECT id FROM users WHERE email = ? AND id != ?",
      [email, decoded.id]
    );

    if (Array.isArray(existingEmail) && existingEmail.length > 0) {
      return NextResponse.json(
        {
          success: false,
          message: "Bu email adresi zaten kullanılıyor",
        },
        { status: 400 }
      );
    }

    // Telefon numarasını temizle (sadece rakamlar)
    const cleanTelefon = telefon ? telefon.replace(/\D/g, "") : "";

    // Şifre güncelleme
    if (sifre && sifre.trim() !== "") {
      // Yeni şifre hash'le
      const hashedPassword = await hashPassword(sifre);
      
      await query(
        `UPDATE users SET 
          username = ?,
          email = ?,
          full_name = ?,
          telefon = ?,
          password = ?
        WHERE id = ?`,
        [kullanici_adi, email, ad_soyad, cleanTelefon, hashedPassword, decoded.id]
      );
    } else {
      // Şifre değiştirilmiyor
      await query(
        `UPDATE users SET 
          username = ?,
          email = ?,
          full_name = ?,
          telefon = ?
        WHERE id = ?`,
        [kullanici_adi, email, ad_soyad, cleanTelefon, decoded.id]
      );
    }

    return NextResponse.json(
      {
        success: true,
        message: "Yönetici bilgileri başarıyla güncellendi",
      },
      { status: 200 }
    );
  } catch (error: any) {
    console.error("Yönetici bilgileri güncelleme hatası:", error);
    return NextResponse.json(
      {
        success: false,
        message: "Yönetici bilgileri güncellenemedi",
        error: error.message,
      },
      { status: 500 }
    );
  }
}

