import { NextRequest, NextResponse } from "next/server";
import { query } from "@/lib/db";
import nodemailer from "nodemailer";

// SMTP bağlantısını test et
export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const {
      smtp_host,
      smtp_port,
      smtp_kullanici_adi,
      smtp_sifre,
      smtp_guvenlik,
      gonderici_email,
      gonderici_adi,
      test_email,
    } = body;

    // Validasyon
    if (!smtp_host || smtp_host.trim() === "") {
      return NextResponse.json(
        {
          success: false,
          message: "SMTP Host zorunludur",
        },
        { status: 400 }
      );
    }

    if (!smtp_port || smtp_port < 1 || smtp_port > 65535) {
      return NextResponse.json(
        {
          success: false,
          message: "Geçerli bir SMTP Port giriniz",
        },
        { status: 400 }
      );
    }

    if (!gonderici_email || gonderici_email.trim() === "") {
      return NextResponse.json(
        {
          success: false,
          message: "Gönderici Email zorunludur",
        },
        { status: 400 }
      );
    }

    // Test emaili kontrolü
    const recipientEmail = test_email || gonderici_email;
    if (!recipientEmail || recipientEmail.trim() === "") {
      return NextResponse.json(
        {
          success: false,
          message: "Test emaili gönderilecek adres zorunludur",
        },
        { status: 400 }
      );
    }

    // Email format kontrolü
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(recipientEmail)) {
      return NextResponse.json(
        {
          success: false,
          message: "Geçerli bir email adresi giriniz",
        },
        { status: 400 }
      );
    }

    // Eğer şifre boşsa, veritabanından al
    let finalPassword = smtp_sifre;
    if (!finalPassword || finalPassword.trim() === "") {
      const existing: any = await query("SELECT smtp_sifre FROM smtp_bilgileri ORDER BY id DESC LIMIT 1");
      if (Array.isArray(existing) && existing.length > 0) {
        finalPassword = existing[0].smtp_sifre || "";
      }
    }

    if (!finalPassword || finalPassword.trim() === "") {
      return NextResponse.json(
        {
          success: false,
          message: "SMTP Şifre gereklidir. Lütfen şifreyi giriniz.",
        },
        { status: 400 }
      );
    }

    // Nodemailer transporter oluştur
    const transporterConfig: any = {
      host: smtp_host,
      port: smtp_port,
      secure: smtp_guvenlik === "ssl", // SSL için true, TLS için false
      auth: {
        user: smtp_kullanici_adi || gonderici_email,
        pass: finalPassword,
      },
    };

    // TLS ayarları
    if (smtp_guvenlik === "tls") {
      transporterConfig.requireTLS = true;
      transporterConfig.tls = {
        rejectUnauthorized: false, // Self-signed sertifikalar için
      };
    } else if (smtp_guvenlik === "none") {
      transporterConfig.secure = false;
      transporterConfig.requireTLS = false;
    }

    const transporter = nodemailer.createTransport(transporterConfig);

    // Bağlantıyı test et
    try {
      await transporter.verify();
    } catch (verifyError: any) {
      return NextResponse.json(
        {
          success: false,
          message: `SMTP bağlantı hatası: ${verifyError.message}`,
        },
        { status: 400 }
      );
    }

    // Test emaili gönder
    const testEmail = {
      from: gonderici_adi
        ? `${gonderici_adi} <${gonderici_email}>`
        : gonderici_email,
      to: recipientEmail, // Modal'dan gelen email adresine gönder
      subject: "SMTP Test Emaili - Teknik Servis Yönetim Sistemi",
      html: `
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
          <h2 style="color: #4F46E5;">SMTP Test Emaili</h2>
          <p>Bu bir test emailidir. SMTP ayarlarınız doğru çalışıyor!</p>
          <hr style="border: none; border-top: 1px solid #e5e7eb; margin: 20px 0;">
          <p style="color: #6b7280; font-size: 12px;">
            Gönderim Zamanı: ${new Date().toLocaleString("tr-TR")}
          </p>
        </div>
      `,
      text: `SMTP Test Emaili\n\nBu bir test emailidir. SMTP ayarlarınız doğru çalışıyor!\n\nGönderim Zamanı: ${new Date().toLocaleString("tr-TR")}`,
    };

    try {
      const info = await transporter.sendMail(testEmail);
      
      return NextResponse.json(
        {
          success: true,
          message: `SMTP bağlantısı başarılı! Test emaili gönderildi. (Message ID: ${info.messageId})`,
        },
        { status: 200 }
      );
    } catch (sendError: any) {
      return NextResponse.json(
        {
          success: false,
          message: `Email gönderim hatası: ${sendError.message}`,
        },
        { status: 400 }
      );
    }
  } catch (error: any) {
    console.error("SMTP test hatası:", error);
    return NextResponse.json(
      {
        success: false,
        message: `SMTP testi başarısız: ${error.message}`,
      },
      { status: 500 }
    );
  }
}

