import { NextRequest, NextResponse } from "next/server";
import { query } from "@/lib/db";

// SMTP bilgilerini getir
export async function GET() {
  try {
    const results: any = await query("SELECT * FROM smtp_bilgileri ORDER BY id DESC LIMIT 1");

    if (Array.isArray(results) && results.length > 0) {
      const data = results[0];
      return NextResponse.json(
        {
          success: true,
          data: {
            smtp_host: data.smtp_host || "",
            smtp_port: data.smtp_port || 587,
            smtp_kullanici_adi: data.smtp_kullanici_adi || "",
            smtp_sifre: "", // Şifreyi güvenlik için göndermiyoruz
            smtp_guvenlik: data.smtp_guvenlik || "tls",
            gonderici_email: data.gonderici_email || "",
            gonderici_adi: data.gonderici_adi || "",
          },
        },
        { status: 200 }
      );
    } else {
      // Eğer kayıt yoksa boş bir kayıt döndür
      return NextResponse.json(
        {
          success: true,
          data: {
            smtp_host: "",
            smtp_port: 587,
            smtp_kullanici_adi: "",
            smtp_sifre: "",
            smtp_guvenlik: "tls",
            gonderici_email: "",
            gonderici_adi: "",
          },
        },
        { status: 200 }
      );
    }
  } catch (error: any) {
    console.error("SMTP bilgileri getirme hatası:", error);
    return NextResponse.json(
      {
        success: false,
        message: "SMTP bilgileri alınamadı",
        error: error.message,
      },
      { status: 500 }
    );
  }
}

// SMTP bilgilerini güncelle veya oluştur
export async function PUT(request: NextRequest) {
  try {
    const body = await request.json();
    const {
      smtp_host,
      smtp_port,
      smtp_kullanici_adi,
      smtp_sifre,
      smtp_guvenlik,
      gonderici_email,
      gonderici_adi,
    } = body;

    // Validasyon
    if (!smtp_host || smtp_host.trim() === "") {
      return NextResponse.json(
        {
          success: false,
          message: "SMTP Host zorunludur",
        },
        { status: 400 }
      );
    }

    if (!smtp_port || smtp_port < 1 || smtp_port > 65535) {
      return NextResponse.json(
        {
          success: false,
          message: "Geçerli bir SMTP Port giriniz (1-65535)",
        },
        { status: 400 }
      );
    }

    if (!gonderici_email || gonderici_email.trim() === "") {
      return NextResponse.json(
        {
          success: false,
          message: "Gönderici Email zorunludur",
        },
        { status: 400 }
      );
    }

    // Önce mevcut kaydı kontrol et
    const existing: any = await query("SELECT id, smtp_sifre FROM smtp_bilgileri ORDER BY id DESC LIMIT 1");

    if (Array.isArray(existing) && existing.length > 0) {
      // Mevcut kaydı güncelle
      // Şifre boşsa eski şifreyi koru
      let finalPassword = existing[0].smtp_sifre;
      if (smtp_sifre && smtp_sifre.trim() !== "") {
        finalPassword = smtp_sifre; // Şifre değiştiriliyor
      }

      await query(
        `UPDATE smtp_bilgileri SET 
          smtp_host = ?,
          smtp_port = ?,
          smtp_kullanici_adi = ?,
          smtp_sifre = ?,
          smtp_guvenlik = ?,
          gonderici_email = ?,
          gonderici_adi = ?
        WHERE id = ?`,
        [
          smtp_host,
          smtp_port,
          smtp_kullanici_adi || "",
          finalPassword,
          smtp_guvenlik || "tls",
          gonderici_email,
          gonderici_adi || "",
          existing[0].id,
        ]
      );

      return NextResponse.json(
        {
          success: true,
          message: "SMTP bilgileri başarıyla güncellendi",
        },
        { status: 200 }
      );
    } else {
      // Yeni kayıt oluştur
      if (!smtp_sifre || smtp_sifre.trim() === "") {
        return NextResponse.json(
          {
            success: false,
            message: "İlk kayıt için SMTP Şifre zorunludur",
          },
          { status: 400 }
        );
      }

      await query(
        `INSERT INTO smtp_bilgileri (
          smtp_host, smtp_port, smtp_kullanici_adi, smtp_sifre,
          smtp_guvenlik, gonderici_email, gonderici_adi
        ) VALUES (?, ?, ?, ?, ?, ?, ?)`,
        [
          smtp_host,
          smtp_port,
          smtp_kullanici_adi || "",
          smtp_sifre,
          smtp_guvenlik || "tls",
          gonderici_email,
          gonderici_adi || "",
        ]
      );

      return NextResponse.json(
        {
          success: true,
          message: "SMTP bilgileri başarıyla kaydedildi",
        },
        { status: 201 }
      );
    }
  } catch (error: any) {
    console.error("SMTP bilgileri güncelleme hatası:", error);
    return NextResponse.json(
      {
        success: false,
        message: "SMTP bilgileri güncellenemedi",
        error: error.message,
      },
      { status: 500 }
    );
  }
}

