import { NextRequest, NextResponse } from "next/server";
import { query } from "@/lib/db";

// Vatan SMS API testi - Kredi sorgulama veya test SMS gönderme
export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const { test_type, test_phone } = body;

    // Veritabanından SMS ayarlarını al
    const smsSettings: any = await query("SELECT api_id, api_key, sms_baslik, api_url FROM sms_ayarlari ORDER BY id DESC LIMIT 1");
    
    if (!Array.isArray(smsSettings) || smsSettings.length === 0) {
      return NextResponse.json(
        {
          success: false,
          message: "SMS ayarları bulunamadı. Lütfen önce SMS ayarlarını kaydedin.",
        },
        { status: 400 }
      );
    }

    const settings = smsSettings[0];
    const api_id = settings.api_id;
    const finalApiKey = settings.api_key;
    const sms_baslik = settings.sms_baslik || "";
    const apiUrl = settings.api_url || "https://api.vatansms.net/api/v1";

    // Validasyon
    if (!api_id || api_id.trim() === "") {
      return NextResponse.json(
        {
          success: false,
          message: "Veritabanında API ID bulunamadı. Lütfen SMS ayarlarını güncelleyin.",
        },
        { status: 400 }
      );
    }

    if (!finalApiKey || finalApiKey.trim() === "") {
      return NextResponse.json(
        {
          success: false,
          message: "Veritabanında API Key bulunamadı. Lütfen SMS ayarlarını güncelleyin.",
        },
        { status: 400 }
      );
    }

    // Test tipine göre işlem yap
    if (test_type === "user_info") {
      // Kullanıcı Bilgileri Sorgulama
      try {
        // API URL'ini temizle ve doğru formatla
        let cleanApiUrl = apiUrl.trim();
        if (cleanApiUrl.endsWith("/")) {
          cleanApiUrl = cleanApiUrl.slice(0, -1);
        }
        // Eğer URL'de endpoint varsa temizle
        if (cleanApiUrl.includes("/1toN") || cleanApiUrl.includes("/balance") || cleanApiUrl.includes("/senders") || cleanApiUrl.includes("/user")) {
          cleanApiUrl = cleanApiUrl.split("/").slice(0, -1).join("/");
        }
        const userInfoUrl = `${cleanApiUrl}/user/information`;
        
        const userInfoData = {
          api_id: api_id,
          api_key: finalApiKey,
        };

        const response = await fetch(userInfoUrl, {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
          },
          body: JSON.stringify(userInfoData),
        });

        // Yanıt içeriğini text olarak al
        const responseText = await response.text();
        
        // Debug bilgisi
        console.log("Kullanıcı Bilgileri Sorgulama Yanıtı:", {
          status: response.status,
          statusText: response.statusText,
          contentType: response.headers.get("content-type"),
          url: userInfoUrl,
          responseLength: responseText.length,
          responsePreview: responseText.substring(0, 500),
        });
        
        // JSON parse etmeyi dene
        let responseData;
        try {
          responseData = JSON.parse(responseText);
        } catch (parseError) {
          // HTML yanıtı geldiğinde veya başka bir format
          console.error("API yanıtı JSON değil (User Info):", {
            status: response.status,
            statusText: response.statusText,
            contentType: response.headers.get("content-type"),
            responsePreview: responseText.substring(0, 500),
          });
          
          // Eğer yanıt boş değilse ve HTML içeriyorsa
          if (responseText.trim().startsWith("<!DOCTYPE") || responseText.trim().startsWith("<html")) {
            return NextResponse.json(
              {
                success: false,
                message: "API'den HTML yanıtı alındı. Endpoint veya API bilgilerinizi kontrol edin.",
                error: "HTML yanıtı alındı",
                debug: {
                  status: response.status,
                  url: userInfoUrl,
                  responsePreview: responseText.substring(0, 300),
                },
              },
              { status: 400 }
            );
          }
          
          // Eğer sadece text yanıtı varsa
          return NextResponse.json(
            {
              success: false,
              message: `API yanıtı: ${responseText.substring(0, 200)}`,
              error: "JSON olmayan yanıt alındı",
              debug: {
                status: response.status,
                url: userInfoUrl,
                rawResponse: responseText,
              },
            },
            { status: 400 }
          );
        }

        if (response.ok) {
          // Kullanıcı bilgileri başarıyla alındı
          const userInfo = responseData.data || responseData;
          const balance = userInfo.balance || userInfo.credit || userInfo.amount || "Bilinmiyor";
          const username = userInfo.username || userInfo.name || "Bilinmiyor";
          const email = userInfo.email || "Bilinmiyor";
          
          return NextResponse.json(
            {
              success: true,
              message: `Kullanıcı bilgileri başarıyla sorgulandı! Kullanıcı: ${username}, Bakiye: ${balance}`,
              data: {
                username: username,
                email: email,
                balance: balance,
                ...userInfo,
              },
            },
            { status: 200 }
          );
        } else {
          return NextResponse.json(
            {
              success: false,
              message: responseData.message || "Kullanıcı bilgileri sorgulanamadı",
              data: responseData,
            },
            { status: 400 }
          );
        }
      } catch (fetchError: any) {
        return NextResponse.json(
          {
            success: false,
            message: `API bağlantı hatası: ${fetchError.message}`,
          },
          { status: 500 }
        );
      }
    } else if (test_type === "senders" || test_type === "originator") {
      // Gönderici Adı Sorgulama
      try {
        // API URL'ini temizle ve doğru formatla
        let cleanApiUrl = apiUrl.trim();
        if (cleanApiUrl.endsWith("/")) {
          cleanApiUrl = cleanApiUrl.slice(0, -1);
        }
        // Eğer URL'de endpoint varsa temizle
        if (cleanApiUrl.includes("/1toN") || cleanApiUrl.includes("/balance") || cleanApiUrl.includes("/user-info")) {
          cleanApiUrl = cleanApiUrl.split("/").slice(0, -1).join("/");
        }
        const sendersUrl = `${cleanApiUrl}/senders`;
        
        const sendersData = {
          api_id: api_id,
          api_key: finalApiKey,
        };

        const response = await fetch(sendersUrl, {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
          },
          body: JSON.stringify(sendersData),
        });

        // Yanıt içeriğini text olarak al
        const responseText = await response.text();
        
        // Debug bilgisi
        console.log("Gönderici Adı Sorgulama Yanıtı:", {
          status: response.status,
          statusText: response.statusText,
          contentType: response.headers.get("content-type"),
          url: sendersUrl,
          responseLength: responseText.length,
          responsePreview: responseText.substring(0, 500),
        });
        
        // JSON parse etmeyi dene
        let responseData;
        try {
          responseData = JSON.parse(responseText);
        } catch (parseError) {
          // HTML yanıtı geldiğinde veya başka bir format
          console.error("API yanıtı JSON değil (Senders):", {
            status: response.status,
            statusText: response.statusText,
            contentType: response.headers.get("content-type"),
            responsePreview: responseText.substring(0, 500),
          });
          
          // Eğer yanıt boş değilse ve HTML içeriyorsa
          if (responseText.trim().startsWith("<!DOCTYPE") || responseText.trim().startsWith("<html")) {
            return NextResponse.json(
              {
                success: false,
                message: "API'den HTML yanıtı alındı. Endpoint veya API bilgilerinizi kontrol edin.",
                error: "HTML yanıtı alındı",
                debug: {
                  status: response.status,
                  url: sendersUrl,
                  responsePreview: responseText.substring(0, 300),
                },
              },
              { status: 400 }
            );
          }
          
          // Eğer sadece text yanıtı varsa (JSON değil ama hata mesajı olabilir)
          return NextResponse.json(
            {
              success: false,
              message: `API yanıtı: ${responseText.substring(0, 200)}`,
              error: "JSON olmayan yanıt alındı",
              debug: {
                status: response.status,
                url: sendersUrl,
                rawResponse: responseText,
              },
            },
            { status: 400 }
          );
        }

        if (response.ok) {
          // Gönderici adları başarıyla alındı
          const senders = responseData.senders || responseData.data || responseData.originators || [];
          const sendersList = Array.isArray(senders) ? senders : [];
          
          if (sendersList.length > 0) {
            return NextResponse.json(
              {
                success: true,
                message: `Gönderici adları başarıyla sorgulandı! Toplam ${sendersList.length} adet gönderici adı bulundu: ${sendersList.join(", ")}`,
                data: {
                  senders: sendersList,
                  count: sendersList.length,
                  ...responseData,
                },
              },
              { status: 200 }
            );
          } else {
            return NextResponse.json(
              {
                success: true,
                message: "Gönderici adı sorgulama başarılı! Ancak henüz onaylanmış gönderici adı bulunmamaktadır.",
                data: {
                  senders: [],
                  count: 0,
                  ...responseData,
                },
              },
              { status: 200 }
            );
          }
        } else {
          return NextResponse.json(
            {
              success: false,
              message: responseData.message || "Gönderici adları sorgulanamadı",
              data: responseData,
            },
            { status: 400 }
          );
        }
      } catch (fetchError: any) {
        return NextResponse.json(
          {
            success: false,
            message: `API bağlantı hatası: ${fetchError.message}`,
          },
          { status: 500 }
        );
      }
    } else if (test_type === "balance" || test_type === "credit") {
      // Bakiye sorgulama
      try {
        // API URL'ini temizle ve doğru formatla
        let cleanApiUrl = apiUrl.trim();
        if (cleanApiUrl.endsWith("/")) {
          cleanApiUrl = cleanApiUrl.slice(0, -1);
        }
        // Eğer URL'de endpoint varsa temizle
        if (cleanApiUrl.includes("/1toN") || cleanApiUrl.includes("/senders") || cleanApiUrl.includes("/user-info")) {
          cleanApiUrl = cleanApiUrl.split("/").slice(0, -1).join("/");
        }
        const balanceUrl = `${cleanApiUrl}/balance`;
        
        const balanceData = {
          api_id: api_id,
          api_key: finalApiKey,
        };

        const response = await fetch(balanceUrl, {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
          },
          body: JSON.stringify(balanceData),
        });

        // Yanıt içeriğini text olarak al
        const responseText = await response.text();
        
        // JSON parse etmeyi dene
        let responseData;
        try {
          responseData = JSON.parse(responseText);
        } catch (parseError) {
          // HTML yanıtı geldiğinde veya başka bir format
          console.error("API yanıtı JSON değil:", {
            status: response.status,
            statusText: response.statusText,
            contentType: response.headers.get("content-type"),
            responsePreview: responseText.substring(0, 500),
          });
          
          // Eğer yanıt boş değilse ve HTML içeriyorsa
          if (responseText.trim().startsWith("<!DOCTYPE") || responseText.trim().startsWith("<html")) {
            return NextResponse.json(
              {
                success: false,
                message: "API'den HTML yanıtı alındı. Endpoint veya API bilgilerinizi kontrol edin.",
                error: "HTML yanıtı alındı",
                debug: {
                  status: response.status,
                  url: balanceUrl,
                  responsePreview: responseText.substring(0, 300),
                },
              },
              { status: 400 }
            );
          }
          
          // Eğer sadece text yanıtı varsa (JSON değil ama hata mesajı olabilir)
          return NextResponse.json(
            {
              success: false,
              message: `API yanıtı: ${responseText.substring(0, 200)}`,
              error: "JSON olmayan yanıt alındı",
              debug: {
                status: response.status,
                url: balanceUrl,
                rawResponse: responseText,
              },
            },
            { status: 400 }
          );
        }

        if (response.ok) {
          // Bakiye bilgisi başarıyla alındı
          const balance = responseData.balance || responseData.credit || responseData.amount || responseData.data?.balance || "Bilinmiyor";
          return NextResponse.json(
            {
              success: true,
              message: `Bakiye sorgulama başarılı! Mevcut bakiyeniz: ${balance}`,
              data: {
                balance: balance,
                ...responseData,
              },
            },
            { status: 200 }
          );
        } else {
          // Alternatif endpoint denemesi: GET metodu ile
          try {
            // API URL'ini temizle ve doğru formatla
            let cleanApiUrl = apiUrl.trim();
            if (cleanApiUrl.endsWith("/")) {
              cleanApiUrl = cleanApiUrl.slice(0, -1);
            }
            // Eğer URL'de endpoint varsa temizle
            if (cleanApiUrl.includes("/1toN") || cleanApiUrl.includes("/senders") || cleanApiUrl.includes("/user-info")) {
              cleanApiUrl = cleanApiUrl.split("/").slice(0, -1).join("/");
            }
            const getBalanceUrl = `${cleanApiUrl}/balance?api_id=${encodeURIComponent(api_id)}&api_key=${encodeURIComponent(finalApiKey)}`;
            const getResponse = await fetch(getBalanceUrl, {
              method: "GET",
              headers: {
                "Content-Type": "application/json",
              },
            });

            // Yanıt içeriğini text olarak al
            const getResponseText = await getResponse.text();
            
            // JSON parse etmeyi dene
            let getResponseData;
            try {
              getResponseData = JSON.parse(getResponseText);
            } catch (parseError) {
              // HTML yanıtı geldiğinde veya başka bir format
              console.error("API yanıtı JSON değil (GET):", {
                status: getResponse.status,
                statusText: getResponse.statusText,
                contentType: getResponse.headers.get("content-type"),
                responsePreview: getResponseText.substring(0, 500),
              });
              
              // Eğer yanıt boş değilse ve HTML içeriyorsa
              if (getResponseText.trim().startsWith("<!DOCTYPE") || getResponseText.trim().startsWith("<html")) {
                return NextResponse.json(
                  {
                    success: false,
                    message: "API'den HTML yanıtı alındı. Endpoint veya API bilgilerinizi kontrol edin.",
                    error: "HTML yanıtı alındı",
                    debug: {
                      status: getResponse.status,
                      url: getBalanceUrl,
                      responsePreview: getResponseText.substring(0, 300),
                    },
                  },
                  { status: 400 }
                );
              }
              
              // Eğer sadece text yanıtı varsa
              return NextResponse.json(
                {
                  success: false,
                  message: `API yanıtı: ${getResponseText.substring(0, 200)}`,
                  error: "JSON olmayan yanıt alındı",
                  debug: {
                    status: getResponse.status,
                    url: getBalanceUrl,
                    rawResponse: getResponseText,
                  },
                },
                { status: 400 }
              );
            }

            if (getResponse.ok) {
              const balance = getResponseData.balance || getResponseData.credit || getResponseData.amount || getResponseData.data?.balance || "Bilinmiyor";
              return NextResponse.json(
                {
                  success: true,
                  message: `Bakiye sorgulama başarılı! Mevcut bakiyeniz: ${balance}`,
                  data: {
                    balance: balance,
                    ...getResponseData,
                  },
                },
                { status: 200 }
              );
            } else {
              return NextResponse.json(
                {
                  success: false,
                  message: getResponseData.message || "Bakiye sorgulanamadı",
                  data: getResponseData,
                },
                { status: 400 }
              );
            }
          } catch (getError: any) {
            return NextResponse.json(
              {
                success: false,
                message: `Bakiye sorgulama hatası: ${responseData.message || getError.message || "API endpoint'i bulunamadı"}`,
                data: responseData,
              },
              { status: 400 }
            );
          }
        }
      } catch (fetchError: any) {
        return NextResponse.json(
          {
            success: false,
            message: `API bağlantı hatası: ${fetchError.message}`,
          },
          { status: 500 }
        );
      }
    } else if (test_type === "test_sms") {
      // Test SMS gönderme
      if (!test_phone || test_phone.trim() === "") {
        return NextResponse.json(
          {
            success: false,
            message: "Test telefon numarası zorunludur",
          },
          { status: 400 }
        );
      }

      // SMS Başlığı opsiyonel - boşsa varsayılan değer kullanılabilir
      const finalSmsBaslik = sms_baslik && sms_baslik.trim() !== "" ? sms_baslik : "SMS";

      // Telefon numarasını temizle (sadece rakamlar)
      const cleanPhone = test_phone.replace(/\D/g, "");
      if (cleanPhone.length < 10) {
        return NextResponse.json(
          {
            success: false,
            message: "Geçerli bir telefon numarası giriniz",
          },
          { status: 400 }
        );
      }

      // Vatan SMS API formatına göre telefon numarasını düzenle
      let formattedPhone = cleanPhone;
      if (!formattedPhone.startsWith("90") && formattedPhone.length === 10) {
        formattedPhone = "90" + formattedPhone;
      }

      // Vatan SMS API isteği
      const smsData = {
        api_id: api_id,
        api_key: finalApiKey,
        sender: finalSmsBaslik,
        message_type: "normal",
        message: "Bu bir test mesajıdır. SMS ayarlarınız doğru çalışıyor!",
        message_content_type: "bilgi",
        phones: [formattedPhone],
      };

      try {
        // API URL'ini temizle ve doğru formatla
        let cleanApiUrl = apiUrl.trim();
        if (cleanApiUrl.endsWith("/")) {
          cleanApiUrl = cleanApiUrl.slice(0, -1);
        }
        // Eğer URL'de endpoint varsa temizle
        if (cleanApiUrl.includes("/1toN") || cleanApiUrl.includes("/balance") || cleanApiUrl.includes("/senders") || cleanApiUrl.includes("/user") || cleanApiUrl.includes("/user-info")) {
          cleanApiUrl = cleanApiUrl.split("/").slice(0, -1).join("/");
        }
        const testSmsUrl = `${cleanApiUrl}/1toN`;
        
        // Debug bilgisi
        console.log("Test SMS Gönderme İsteği:", {
          url: testSmsUrl,
          api_id: api_id,
          sender: finalSmsBaslik,
          phone: formattedPhone,
        });

        const response = await fetch(testSmsUrl, {
          method: "POST",
          headers: {
            "Content-Type": "application/json",
          },
          body: JSON.stringify(smsData),
        });

        // Yanıt içeriğini text olarak al
        const responseText = await response.text();
        
        // Debug bilgisi
        console.log("Test SMS Gönderme Yanıtı:", {
          status: response.status,
          statusText: response.statusText,
          contentType: response.headers.get("content-type"),
          url: testSmsUrl,
          responseLength: responseText.length,
          responsePreview: responseText.substring(0, 500),
        });
        
        // JSON parse etmeyi dene
        let responseData;
        try {
          responseData = JSON.parse(responseText);
        } catch (parseError) {
          // HTML yanıtı geldiğinde veya başka bir format
          console.error("API yanıtı JSON değil (Test SMS):", {
            status: response.status,
            statusText: response.statusText,
            contentType: response.headers.get("content-type"),
            responsePreview: responseText.substring(0, 500),
          });
          
          // Eğer yanıt boş değilse ve HTML içeriyorsa
          if (responseText.trim().startsWith("<!DOCTYPE") || responseText.trim().startsWith("<html")) {
            return NextResponse.json(
              {
                success: false,
                message: "API'den HTML yanıtı alındı. Endpoint veya API bilgilerinizi kontrol edin.",
                error: "HTML yanıtı alındı",
                debug: {
                  status: response.status,
                  url: testSmsUrl,
                  responsePreview: responseText.substring(0, 300),
                },
              },
              { status: 400 }
            );
          }
          
          // Eğer sadece text yanıtı varsa
          return NextResponse.json(
            {
              success: false,
              message: `API yanıtı: ${responseText.substring(0, 200)}`,
              error: "JSON olmayan yanıt alındı",
              debug: {
                status: response.status,
                url: testSmsUrl,
                rawResponse: responseText,
              },
            },
            { status: 400 }
          );
        }

        if (response.ok) {
          // Başarılı yanıt kontrolü - farklı API formatları için
          const isSuccess = responseData.status === "success" || 
                           responseData.success === true || 
                           responseData.message_id || 
                           (responseData.data && responseData.data.message_id);
          
          if (isSuccess) {
            const messageId = responseData.message_id || responseData.data?.message_id || "N/A";
            return NextResponse.json(
              {
                success: true,
                message: `Test SMS başarıyla gönderildi! Mesaj ID: ${messageId}`,
                data: responseData,
              },
              { status: 200 }
            );
          } else {
            // API başarılı yanıt verdi ama SMS gönderilemedi
            const errorMessage = responseData.description || responseData.message || responseData.error || "SMS gönderilemedi";
            return NextResponse.json(
              {
                success: false,
                message: errorMessage,
                data: responseData,
              },
              { status: 400 }
            );
          }
        } else {
          // Hata durumunda description alanını öncelikle göster
          const errorMessage = responseData.description || responseData.message || responseData.error || "SMS gönderilemedi";
          return NextResponse.json(
            {
              success: false,
              message: errorMessage,
              data: responseData,
            },
            { status: 400 }
          );
        }
      } catch (fetchError: any) {
        console.error("Test SMS API bağlantı hatası:", fetchError);
        return NextResponse.json(
          {
            success: false,
            message: `API bağlantı hatası: ${fetchError.message}`,
            error: fetchError.message,
          },
          { status: 500 }
        );
      }
    } else {
      return NextResponse.json(
        {
          success: false,
          message: "Geçersiz test tipi",
        },
        { status: 400 }
      );
    }
  } catch (error: any) {
    console.error("SMS test hatası:", error);
    return NextResponse.json(
      {
        success: false,
        message: `SMS testi başarısız: ${error.message}`,
      },
      { status: 500 }
    );
  }
}
