import { NextRequest, NextResponse } from "next/server";
import { query } from "@/lib/db";

// N SMS Gönderimi (1toN - Birden fazla numaraya aynı mesaj)
export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const { message, phones, message_type, message_content_type } = body;

    // Veritabanından SMS ayarlarını al
    const smsSettings: any = await query("SELECT api_id, api_key, sms_baslik, api_url FROM sms_ayarlari ORDER BY id DESC LIMIT 1");
    
    if (!Array.isArray(smsSettings) || smsSettings.length === 0) {
      return NextResponse.json(
        {
          success: false,
          message: "SMS ayarları bulunamadı. Lütfen önce SMS ayarlarını kaydedin.",
        },
        { status: 400 }
      );
    }

    const settings = smsSettings[0];
    const api_id = settings.api_id;
    const finalApiKey = settings.api_key;
    const sms_baslik = settings.sms_baslik || "";
    const apiUrl = settings.api_url || "https://api.vatansms.net/api/v1";

    // Validasyon
    if (!api_id || api_id.trim() === "") {
      return NextResponse.json(
        {
          success: false,
          message: "Veritabanında API ID bulunamadı. Lütfen SMS ayarlarını güncelleyin.",
        },
        { status: 400 }
      );
    }

    if (!finalApiKey || finalApiKey.trim() === "") {
      return NextResponse.json(
        {
          success: false,
          message: "Veritabanında API Key bulunamadı. Lütfen SMS ayarlarını güncelleyin.",
        },
        { status: 400 }
      );
    }

    // SMS Başlığı opsiyonel - boşsa varsayılan değer kullanılabilir
    const finalSmsBaslik = sms_baslik && sms_baslik.trim() !== "" ? sms_baslik : "SMS";

    if (!message || message.trim() === "") {
      return NextResponse.json(
        {
          success: false,
          message: "Mesaj içeriği zorunludur",
        },
        { status: 400 }
      );
    }

    if (!phones || !Array.isArray(phones) || phones.length === 0) {
      return NextResponse.json(
        {
          success: false,
          message: "En az bir telefon numarası gereklidir",
        },
        { status: 400 }
      );
    }

    // Telefon numaralarını temizle ve formatla
    const formattedPhones = phones
      .map((phone: string) => {
        // Sadece rakamları al
        const cleanPhone = phone.replace(/\D/g, "");
        
        // Boş numaraları atla
        if (cleanPhone.length < 10) {
          return null;
        }

        // Vatan SMS API formatına göre telefon numarasını düzenle
        let formattedPhone = cleanPhone;
        if (!formattedPhone.startsWith("90") && formattedPhone.length === 10) {
          formattedPhone = "90" + formattedPhone;
        }

        return formattedPhone;
      })
      .filter((phone: string | null) => phone !== null) as string[];

    if (formattedPhones.length === 0) {
      return NextResponse.json(
        {
          success: false,
          message: "Geçerli telefon numarası bulunamadı",
        },
        { status: 400 }
      );
    }

    // Vatan SMS API isteği - 1toN (N SMS Gönderimi)
    const smsData = {
      api_id: api_id,
      api_key: finalApiKey,
      sender: finalSmsBaslik,
      message_type: message_type || "normal",
      message: message,
      message_content_type: message_content_type || "bilgi",
      phones: formattedPhones,
    };

    try {
      // API URL'ini temizle ve doğru formatla
      let cleanApiUrl = apiUrl.trim();
      if (cleanApiUrl.endsWith("/")) {
        cleanApiUrl = cleanApiUrl.slice(0, -1);
      }
      // Eğer URL'de endpoint varsa temizle
      if (cleanApiUrl.includes("/balance") || cleanApiUrl.includes("/senders") || cleanApiUrl.includes("/user-info")) {
        cleanApiUrl = cleanApiUrl.split("/").slice(0, -1).join("/");
      }
      const response = await fetch(`${cleanApiUrl}/1toN`, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify(smsData),
      });

      // Yanıt içeriğini text olarak al
      const responseText = await response.text();
      
      // JSON parse etmeyi dene
      let responseData;
      try {
        responseData = JSON.parse(responseText);
      } catch (parseError) {
        // HTML yanıtı geldiğinde veya başka bir format
        console.error("API yanıtı JSON değil (Send SMS):", {
          status: response.status,
          statusText: response.statusText,
          contentType: response.headers.get("content-type"),
          responsePreview: responseText.substring(0, 500),
        });
        
        // Eğer yanıt boş değilse ve HTML içeriyorsa
        if (responseText.trim().startsWith("<!DOCTYPE") || responseText.trim().startsWith("<html")) {
          return NextResponse.json(
            {
              success: false,
              message: "API'den HTML yanıtı alındı. Endpoint veya API bilgilerinizi kontrol edin.",
              error: "HTML yanıtı alındı",
              debug: {
                status: response.status,
                url: `${apiUrl}/1toN`,
                responsePreview: responseText.substring(0, 300),
              },
            },
            { status: 400 }
          );
        }
        
        // Eğer sadece text yanıtı varsa
        return NextResponse.json(
          {
            success: false,
            message: `API yanıtı: ${responseText.substring(0, 200)}`,
            error: "JSON olmayan yanıt alındı",
            debug: {
              status: response.status,
              url: `${apiUrl}/1toN`,
              rawResponse: responseText,
            },
          },
          { status: 400 }
        );
      }

      if (response.ok && responseData.status === "success") {
        return NextResponse.json(
          {
            success: true,
            message: `${formattedPhones.length} adet SMS başarıyla gönderildi!`,
            data: {
              sent_count: formattedPhones.length,
              message_id: responseData.message_id,
              ...responseData,
            },
          },
          { status: 200 }
        );
      } else {
        // Hata durumunda description alanını öncelikle göster
        const errorMessage = responseData.description || responseData.message || responseData.error || "SMS gönderilemedi";
        return NextResponse.json(
          {
            success: false,
            message: errorMessage,
            data: responseData,
          },
          { status: 400 }
        );
      }
    } catch (fetchError: any) {
      return NextResponse.json(
        {
          success: false,
          message: `API bağlantı hatası: ${fetchError.message}`,
        },
        { status: 500 }
      );
    }
  } catch (error: any) {
    console.error("N SMS gönderim hatası:", error);
    return NextResponse.json(
      {
        success: false,
        message: `SMS gönderimi başarısız: ${error.message}`,
      },
      { status: 500 }
    );
  }
}

