import { NextRequest, NextResponse } from "next/server";
import { query } from "@/lib/db";

// SMS ayarlarını getir
export async function GET() {
  try {
    const results: any = await query("SELECT * FROM sms_ayarlari ORDER BY id DESC LIMIT 1");

    if (Array.isArray(results) && results.length > 0) {
      const data = results[0];
      return NextResponse.json(
        {
          success: true,
          data: {
            api_id: data.api_id || "",
            api_key: data.api_key || "",
            sms_baslik: data.sms_baslik || "",
            api_url: data.api_url || "https://api.vatansms.net/api/v1",
          },
        },
        { status: 200 }
      );
    } else {
      // Eğer kayıt yoksa boş bir kayıt döndür
      return NextResponse.json(
        {
          success: true,
          data: {
            api_id: "",
            api_key: "",
            sms_baslik: "",
            api_url: "https://api.vatansms.net/api/v1",
          },
        },
        { status: 200 }
      );
    }
  } catch (error: any) {
    console.error("SMS ayarları getirme hatası:", error);
    return NextResponse.json(
      {
        success: false,
        message: "SMS ayarları alınamadı",
        error: error.message,
      },
      { status: 500 }
    );
  }
}

// SMS ayarlarını güncelle veya oluştur
export async function PUT(request: NextRequest) {
  try {
    const body = await request.json();
    const { api_id, api_key, sms_baslik, api_url } = body;

    // Validasyon
    if (!api_id || api_id.trim() === "") {
      return NextResponse.json(
        {
          success: false,
          message: "API ID zorunludur",
        },
        { status: 400 }
      );
    }

    // Önce mevcut kaydı kontrol et
    const existing: any = await query("SELECT id, api_key FROM sms_ayarlari ORDER BY id DESC LIMIT 1");

    if (Array.isArray(existing) && existing.length > 0) {
      // Mevcut kaydı güncelle
      // API Key boşsa eski key'i koru
      let finalApiKey = existing[0].api_key;
      if (api_key && api_key.trim() !== "") {
        finalApiKey = api_key; // API Key değiştiriliyor
      }

      await query(
        `UPDATE sms_ayarlari SET 
          api_id = ?,
          api_key = ?,
          sms_baslik = ?,
          api_url = ?
        WHERE id = ?`,
        [
          api_id,
          finalApiKey,
          sms_baslik || "",
          api_url || "https://api.vatansms.net/api/v1",
          existing[0].id,
        ]
      );

      return NextResponse.json(
        {
          success: true,
          message: "SMS ayarları başarıyla güncellendi",
        },
        { status: 200 }
      );
    } else {
      // Yeni kayıt oluştur
      if (!api_key || api_key.trim() === "") {
        return NextResponse.json(
          {
            success: false,
            message: "İlk kayıt için API Key zorunludur",
          },
          { status: 400 }
        );
      }

      await query(
        `INSERT INTO sms_ayarlari (
          api_id, api_key, sms_baslik, api_url
        ) VALUES (?, ?, ?, ?)`,
        [
          api_id,
          api_key,
          sms_baslik || "",
          api_url || "https://api.vatansms.net/api/v1",
        ]
      );

      return NextResponse.json(
        {
          success: true,
          message: "SMS ayarları başarıyla kaydedildi",
        },
        { status: 201 }
      );
    }
  } catch (error: any) {
    console.error("SMS ayarları güncelleme hatası:", error);
    return NextResponse.json(
      {
        success: false,
        message: "SMS ayarları güncellenemedi",
        error: error.message,
      },
      { status: 500 }
    );
  }
}

