import { NextRequest, NextResponse } from "next/server";
import { query } from "@/lib/db";
import { writeFile, mkdir } from "fs/promises";
import { join } from "path";
import { existsSync } from "fs";

// Panel ayarlarını getir
export async function GET() {
  try {
    const results: any = await query("SELECT * FROM panel_ayarlari ORDER BY id DESC LIMIT 1");

    if (Array.isArray(results) && results.length > 0) {
      const data = results[0];
      return NextResponse.json(
        {
          success: true,
          data: {
            panel_adi: data.panel_adi || "",
            logo: data.logo || "",
            stok_uyari_limiti: data.stok_uyari_limiti || "",
            izin_verilen_dosya_turleri: data.izin_verilen_dosya_turleri || "",
            hizmet_kdv_orani: data.hizmet_kdv_orani || "",
            urun_kdv_orani: data.urun_kdv_orani || "",
            para_birimi: data.para_birimi || "TRY",
          },
        },
        { status: 200 }
      );
    } else {
      // Eğer kayıt yoksa boş bir kayıt döndür
      return NextResponse.json(
        {
          success: true,
          data: {
            panel_adi: "",
            logo: "",
            stok_uyari_limiti: "",
            izin_verilen_dosya_turleri: "",
            hizmet_kdv_orani: "",
            urun_kdv_orani: "",
            para_birimi: "TRY",
          },
        },
        { status: 200 }
      );
    }
  } catch (error: any) {
    console.error("Panel ayarları getirme hatası:", error);
    return NextResponse.json(
      {
        success: false,
        message: "Panel ayarları alınamadı",
        error: error.message,
      },
      { status: 500 }
    );
  }
}

// Panel ayarlarını güncelle veya oluştur
export async function PUT(request: NextRequest) {
  try {
    const formData = await request.formData();

    const panel_adi = formData.get("panel_adi") as string;
    const stok_uyari_limiti = formData.get("stok_uyari_limiti") as string;
    const izin_verilen_dosya_turleri = formData.get("izin_verilen_dosya_turleri") as string;
    const hizmet_kdv_orani = formData.get("hizmet_kdv_orani") as string;
    const urun_kdv_orani = formData.get("urun_kdv_orani") as string;
    const para_birimi = formData.get("para_birimi") as string;
    const logo_file = formData.get("logo_file") as File | null;
    const existing_logo = formData.get("logo") as string | null;

    // Validasyon
    if (!panel_adi || panel_adi.trim() === "") {
      return NextResponse.json(
        {
          success: false,
          message: "Panel adı zorunludur",
        },
        { status: 400 }
      );
    }

    if (!para_birimi || para_birimi.trim() === "") {
      return NextResponse.json(
        {
          success: false,
          message: "Para birimi zorunludur",
        },
        { status: 400 }
      );
    }

    // Logo dosyası yükleme işlemi
    let logoPath = existing_logo || "";

    if (logo_file && logo_file.size > 0) {
      // Dosya türü kontrolü
      const allowedTypes = ["image/jpeg", "image/jpg", "image/png", "image/gif", "image/svg+xml"];
      if (!allowedTypes.includes(logo_file.type)) {
        return NextResponse.json(
          {
            success: false,
            message: "Sadece resim dosyaları yüklenebilir (JPEG, PNG, GIF, SVG)",
          },
          { status: 400 }
        );
      }

      // Dosya boyutu kontrolü (5MB)
      if (logo_file.size > 5 * 1024 * 1024) {
        return NextResponse.json(
          {
            success: false,
            message: "Dosya boyutu 5MB'dan küçük olmalıdır",
          },
          { status: 400 }
        );
      }

      // Uploads klasörünü oluştur
      const uploadsDir = join(process.cwd(), "public", "uploads", "logo");
      if (!existsSync(uploadsDir)) {
        await mkdir(uploadsDir, { recursive: true });
      }

      // Dosya adını oluştur (timestamp + orijinal ad)
      const timestamp = Date.now();
      const fileExtension = logo_file.name.split(".").pop();
      const fileName = `logo_${timestamp}.${fileExtension}`;
      const filePath = join(uploadsDir, fileName);

      // Dosyayı kaydet
      const bytes = await logo_file.arrayBuffer();
      const buffer = Buffer.from(bytes);
      await writeFile(filePath, buffer);

      // Veritabanına kaydedilecek yol
      logoPath = `/uploads/logo/${fileName}`;

      // Eski logo dosyasını sil (varsa ve yeni dosya yükleniyorsa)
      if (existing_logo && existing_logo.startsWith("/uploads/logo/")) {
        const oldFilePath = join(process.cwd(), "public", existing_logo);
        if (existsSync(oldFilePath)) {
          try {
            const { unlink } = require("fs/promises");
            await unlink(oldFilePath);
          } catch (err) {
            console.error("Eski logo dosyası silinemedi:", err);
          }
        }
      }
    }

    // Önce mevcut kaydı kontrol et
    const existing: any = await query("SELECT id FROM panel_ayarlari ORDER BY id DESC LIMIT 1");

    // Sayısal değerleri dönüştür
    const stokLimit = stok_uyari_limiti ? parseInt(stok_uyari_limiti) : 0;
    const hizmetKdv = hizmet_kdv_orani ? parseFloat(hizmet_kdv_orani) : 0;
    const urunKdv = urun_kdv_orani ? parseFloat(urun_kdv_orani) : 0;

    if (Array.isArray(existing) && existing.length > 0) {
      // Mevcut kaydı güncelle
      await query(
        `UPDATE panel_ayarlari SET 
          panel_adi = ?,
          logo = ?,
          stok_uyari_limiti = ?,
          izin_verilen_dosya_turleri = ?,
          hizmet_kdv_orani = ?,
          urun_kdv_orani = ?,
          para_birimi = ?
        WHERE id = ?`,
        [
          panel_adi,
          logoPath,
          stokLimit,
          izin_verilen_dosya_turleri || "",
          hizmetKdv,
          urunKdv,
          para_birimi,
          existing[0].id,
        ]
      );

      return NextResponse.json(
        {
          success: true,
          message: "Panel ayarları başarıyla güncellendi",
        },
        { status: 200 }
      );
    } else {
      // Yeni kayıt oluştur
      await query(
        `INSERT INTO panel_ayarlari (
          panel_adi, logo, stok_uyari_limiti, izin_verilen_dosya_turleri,
          hizmet_kdv_orani, urun_kdv_orani, para_birimi
        ) VALUES (?, ?, ?, ?, ?, ?, ?)`,
        [
          panel_adi,
          logoPath,
          stokLimit,
          izin_verilen_dosya_turleri || "",
          hizmetKdv,
          urunKdv,
          para_birimi,
        ]
      );

      return NextResponse.json(
        {
          success: true,
          message: "Panel ayarları başarıyla kaydedildi",
        },
        { status: 201 }
      );
    }
  } catch (error: any) {
    console.error("Panel ayarları güncelleme hatası:", error);
    return NextResponse.json(
      {
        success: false,
        message: "Panel ayarları güncellenemedi",
        error: error.message,
      },
      { status: 500 }
    );
  }
}


