import { NextRequest, NextResponse } from "next/server";
import { verifyToken } from "@/lib/auth";
import { findUserById } from "@/lib/auth";

export async function GET(request: NextRequest) {
  try {
    // Token'ı cookie'den veya header'dan al
    const token =
      request.cookies.get("token")?.value ||
      request.headers.get("authorization")?.replace("Bearer ", "");

    if (!token) {
      return NextResponse.json(
        {
          success: false,
          message: "Token bulunamadı",
        },
        { status: 401 }
      );
    }

    // Token'ı doğrula
    const decoded = verifyToken(token);

    if (!decoded || !decoded.id) {
      return NextResponse.json(
        {
          success: false,
          message: "Geçersiz token",
        },
        { status: 401 }
      );
    }

    // Kullanıcıyı bul
    const user = await findUserById(decoded.id);

    if (!user) {
      return NextResponse.json(
        {
          success: false,
          message: "Kullanıcı bulunamadı",
        },
        { status: 404 }
      );
    }

    return NextResponse.json(
      {
        success: true,
        user: {
          id: user.id,
          username: user.username,
          email: user.email,
          full_name: user.full_name,
          role: user.role,
        },
      },
      { status: 200 }
    );
  } catch (error: any) {
    console.error("Auth check hatası:", error);
    return NextResponse.json(
      {
        success: false,
        message: "Sunucu hatası",
        error: error.message,
      },
      { status: 500 }
    );
  }
}

