import { NextRequest, NextResponse } from "next/server";
import { login } from "@/lib/auth";

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const { username, password } = body;

    // Validasyon
    if (!username || !password) {
      return NextResponse.json(
        {
          success: false,
          message: "Kullanıcı adı ve şifre gereklidir",
        },
        { status: 400 }
      );
    }

    // Login işlemi
    const result = await login({ username, password });

    if (!result) {
      return NextResponse.json(
        {
          success: false,
          message: "Kullanıcı adı veya şifre hatalı",
        },
        { status: 401 }
      );
    }

    // Başarılı login
    const response = NextResponse.json(
      {
        success: true,
        message: "Giriş başarılı",
        user: {
          id: result.user.id,
          username: result.user.username,
          email: result.user.email,
          full_name: result.user.full_name,
          role: result.user.role,
        },
        token: result.token,
      },
      { status: 200 }
    );

    // Cookie'ye token ekle (opsiyonel)
    response.cookies.set("token", result.token, {
      httpOnly: true,
      secure: process.env.NODE_ENV === "production",
      sameSite: "lax",
      maxAge: 60 * 60 * 24 * 7, // 7 gün
      path: "/",
    });

    return response;
  } catch (error: any) {
    console.error("Login API hatası:", error);
    return NextResponse.json(
      {
        success: false,
        message: "Sunucu hatası",
        error: error.message,
      },
      { status: 500 }
    );
  }
}

